/**
* \file: WaylandContext.h
*
* \version: $Id:$
*
* \release: $Name:$
*
* <brief description>.
* <detailed description>
* \component: Digital iPod Out - Wayland Adapter
*
* \author: J. Harder / ADIT/SW1 / jharder@de.adit-jv.com
*
* \copyright (c) 2013 Advanced Driver Information Technology.
* This code is developed by Advanced Driver Information Technology.
* Copyright of Advanced Driver Information Technology, Bosch, and DENSO.
* All rights reserved.
*
* \see <related items>
*
* \history
*
***********************************************************************/

#ifndef DIPO_WAYLANDCONTEXT_H
#define DIPO_WAYLANDCONTEXT_H

#include <atomic>
#include <memory>
#include <dipo_plugin.h>
#include <wayland-util.h>
#include <wayland-client-protocol.h>
#include <compositor-shim.h>
#include <dipo_macros.h>
#include <string.h>
#include <list>
#include "WaylandTouchInputCommon.h"

namespace adit { namespace carplay
{
class WaylandContext;

class WLSeat
{
// properties

private:
    struct wl_seat*       wlSeat;
    char *                wlSeatName;
    uint32_t              wlSeatID;
    struct wl_pointer*    wlPointer;
    struct wl_touch*      wlTouch;
    class WaylandContext* pParentWLContext;

public:
    WLSeat()
    :wlSeat(NULL)
    ,wlSeatID(0)
    ,wlPointer(NULL)
    ,wlTouch(NULL)
    ,pParentWLContext(NULL)
    {

    }
    ~WLSeat()
    {
        if (NULL != wlSeatName)
        {
            free(wlSeatName);
            wlSeatName = nullptr;
        }
        if (NULL != wlSeat)
        {
            dipo_safe_call(wl_seat_destroy, wlSeat);
        }
        if (NULL != wlPointer)
        {
            dipo_safe_call(wl_pointer_release, wlPointer);
        }
        if (NULL != wlTouch)
        {
            dipo_safe_call(wl_touch_release, wlTouch);
        }
    }

    struct wl_seat* GetWLSeat() const { return wlSeat; }
    char* GetSeatName() {return wlSeatName;}
    uint32_t GetSeatID() { return wlSeatID;}
    struct wl_touch* GetWLTouch() const { return wlTouch; }
    struct wl_pointer* GetWLPointer() const { return wlPointer; }
    WaylandContext* GetParentWlContext() {return pParentWLContext;}

    void SetWLSeat(struct wl_seat* inWlSeat) { wlSeat = inWlSeat; }
    void SetSeatName(const char* inWlSeatName) {wlSeatName = strdup(inWlSeatName);}
    void SetSeatID(uint32_t inWlSeatID){wlSeatID = inWlSeatID;}
    void SetWLTouch(struct wl_touch* inWlTouch) { wlTouch = inWlTouch; }
    void SetWLPointer(struct wl_pointer* inWlPointer) { wlPointer = inWlPointer; }
    void SetParentWlContext(WaylandContext* inWlContext) {pParentWLContext = inWlContext; }
};

class DIPO_WL_HIDDEN_SYMBOL WaylandContext
{
    enum WaylandEvents
    {
        event_shutdown = 0,
        event_input =1
    };
public:
    /* abstract base for register-able device listeners */
    class DeviceListener
    {
    public:
        virtual ~DeviceListener();
    };

    WaylandContext();
    WaylandContext(struct wl_display* inDisplay);
    ~WaylandContext();

    bool Initialize();

    inline struct wl_display* GetDisplay() { return display; }
    inline struct wl_compositor* GetCompositor() { return compositor; }
    inline struct compositor_shim_context* GetAdapterContext() {return adapterContext;}
    inline struct wl_shm* GetShm() { return shm; }
    inline uint32_t GetShmFormat() { return shmFormat; }

    void SetTouchListener(struct wl_touch_listener* inListener,
            std::unique_ptr<DeviceListener> inObj);
    void SetPointerListener(struct wl_pointer_listener* inListener,
            std::unique_ptr<DeviceListener> inObj);

    bool StartInputThread(uint32_t inProcessIntervalUSec);
    void StopInputThread();
    void AddWlSeat(WLSeat* pWlSeat) {
        pthread_mutex_lock(&SeatLock);
        wlSeatList.push_back(pWlSeat);
        pthread_mutex_unlock(&SeatLock);
    }
    std::list<WLSeat*>& GetWLSeatList() { return wlSeatList;}

private:
    std::list<WLSeat*> wlSeatList;
    pthread_mutex_t SeatLock;

    bool ilmInit;
    struct wl_display* display;
    bool   shareWlDisplay;
    struct wl_event_queue* inputQueue;
    struct wl_compositor* compositor;
    struct wl_shm *shm;
    uint32_t shmFormat;
    struct wl_registry* registry;
    struct compositor_shim_context* adapterContext;

    struct wl_touch_listener* touchListener;
    std::unique_ptr<DeviceListener> touchListenerObj;

    struct wl_pointer_listener* pointerListener;
    std::unique_ptr<DeviceListener> pointerListenerObj;


    std::atomic<bool> threadRunning;
    pthread_t threadId;
    uint32_t processIntervalUSec;
    int32_t shutDownEventFd;
    const uint64_t shutDownEvent = 1;


    static void* inputThread(void* inData);

    /* static listener information */
    static struct wl_registry_listener registryListener;
    static struct wl_shm_listener shmListener;
    static struct wl_seat_listener seatListener;
    static bool staticInitialized;

    static void staticInitialize();

    static void onRegistry(void* inMe, struct wl_registry* inRegistry, uint32_t inName,
            const char* inInterface, uint32_t inVersion);
    static void onRegistryRemove(void* inMe, struct wl_registry* inRegistry, uint32_t inName );
    static void onShmFormats(void* inMe, struct wl_shm* inShm,
            uint32_t inFormat);
    static void onSeatCapabilities(void* inMe, struct wl_seat* inSeat, uint32_t inCaps);
    static void onSeatName(void *inMe, struct wl_seat* wl_seat, const char *inName);
    void destroyWaylandContext();
};

} } /* namespace adit { namespace carplay */

#endif /* DIPO_WAYLANDCONTEXT_H */
